<?php declare(strict_types=1);

namespace apexl\ConvertKitHelper;

use WP_User;

class ConvertKitValidator
{
    private const FORBIDDEN_USERNAMES = [
        'craftwithsarah'
    ];

    public const FLAG__REQUIRE_PASSWORD = 1;
    public const FLAG__REQUIRE_USERNAME = 2;

    /** @var ConvertKitUserSubmission $data */
    private $data;
    /** @var WP_User|null $currentUser */
    private $currentUser;
    /** @var int $flags */
    private $flags;

    public static function dataFromPost(): ConvertKitUserSubmission
    {
        return new ConvertKitUserSubmission(
            $_POST['com_firstname'] ?? null,
            $_POST['com_lastname'] ?? null,
            $_POST['com_email'] ?? null,
            $_POST['com_username'] ?? null,
            $_POST['com_password'] ?? null,
            $_POST['c_password'] ?? null,
            $_POST['com_interests'] ?? [],
            $_POST['com_machine'] ?? null
        );
    }

    public function __construct(int $flags = 0)
    {
        $this->flags = $flags;
    }

    /**
     * @throws ConvertKitValidationException
     */
    public function validate(ConvertKitUserSubmission $data, ?WP_User $currentUser = null): ConvertKitUserData
    {
        $this->data = $data;
        $this->currentUser = $currentUser;

        $this->assertNotEmpty();
        $this->assertUsernameIsValid();
        $this->assertEmailIsNotTest();
        $this->assertUsernameIsNotTaken();
        $this->assertEmailIsNotTaken();
        $this->assertFirstAndLastNamesAreDifferent();
        $this->assertInterestsIsArray();
        $this->assertMachineIsYesOrNo();
        if ($this->hasPassword()) {
            $this->assertPasswordIsConfirmed();
        }

        return new ConvertKitUserData(
            $this->data->firstname,
            $this->data->lastname,
            $this->data->email,
            $this->data->password,
            $this->data->username ?? null
        );
    }

    /**
     * @throws ConvertKitValidationException
     */
    private function assertNotEmpty()
    {
        if (!$this->data->firstname) {
            throw new ConvertKitValidationException($this->requiredMessage('First name'));
        }

        if (!$this->data->lastname) {
            throw new ConvertKitValidationException($this->requiredMessage('Last name'));
        }

        if (!$this->data->email) {
            throw new ConvertKitValidationException($this->requiredMessage('Email'));
        }

        if ($this->usernameIsRequired()) {
            if (!$this->data->username) {
                throw new ConvertKitValidationException($this->requiredMessage('Username'));
            }
        }

        if (!$this->data->machine) {
            throw new ConvertKitValidationException($this->requiredMessage('Cutting machine'));
        }

        if ($this->passwordIsRequired()) {
            if (!$this->data->password) {
                throw new ConvertKitValidationException($this->requiredMessage('Password'));
            }
        }
    }

    private function requiredMessage(string $field): string
    {
        return sprintf('%s is required', $field);
    }

    /**
     * @throws ConvertKitValidationException
     */
    private function assertUsernameIsValid()
    {
        if ($this->usernameIsForbidden()) {
            throw new ConvertKitValidationException('Username is invalid');
        }
    }

    private function usernameIsForbidden(): bool
    {
        return in_array($this->data->username, self::FORBIDDEN_USERNAMES);
    }

    /**
     * @throws ConvertKitValidationException
     */
    private function assertEmailIsNotTest()
    {
        if (strpos($this->data->email, 'test') !== false) {
            throw new ConvertKitValidationException('Email is invalid');
        }
    }

    /**
     * @throws ConvertKitValidationException
     */
    private function assertUsernameIsNotTaken(): void
    {
        if ($this->currentUser) {
            if ($this->data->username === $this->currentUser->user_login) {
                return;
            }
        }

        if (username_exists($this->data->username)) {
            throw new ConvertKitValidationException(
                'This username is already registered to a different user. Please choose another'
            );
        }
    }

    /**
     * @throws ConvertKitValidationException
     */
    private function assertPasswordIsConfirmed()
    {
        if ($this->data->password !== $this->data->password_confirm) {
            throw new ConvertKitValidationException('Passwords do not match');
        }
    }

    /**
     * @throws ConvertKitValidationException
     */
    private function assertFirstAndLastNamesAreDifferent()
    {
        if ($this->data->firstname === $this->data->lastname) {
            throw new ConvertKitValidationException('First and Last names must be different');
        }
    }

    /**
     * @throws ConvertKitValidationException
     */
    private function assertEmailIsNotTaken()
    {
        if ($this->currentUser) {
            if ($this->data->email === $this->currentUser->user_email) {
                return;
            }
        }

        if (email_exists($this->data->email)) {
            throw new ConvertKitValidationException(
                'Email already registered. Please <a href=\'/login/\'>log in to your account</a>.'
            );
        }
    }

    private function usernameIsRequired(): bool
    {
        return ($this->flags & self::FLAG__REQUIRE_USERNAME) !== 0;
    }

    private function passwordIsRequired(): bool
    {
        return ($this->flags & self::FLAG__REQUIRE_PASSWORD) !== 0;
    }

    private function hasPassword(): bool
    {
        return !!$this->data->password;
    }

    /**
     * @throws ConvertKitValidationException
     */
    private function assertInterestsIsArray()
    {
        if (!is_array($this->data->interests)) {
            throw new ConvertKitValidationException('Interests invalid');
        }
    }

    /**
     * @throws ConvertKitValidationException
     */
    private function assertMachineIsYesOrNo()
    {
        if (!in_array($this->data->machine, ['yes', 'no'])) {
            throw new ConvertKitValidationException('Machine invalid');
        }
    }

    public function requirePassword(): ConvertKitValidator
    {
        $this->flags = $this->flags | self::FLAG__REQUIRE_PASSWORD;

        return $this;
    }

    public function requireUsername(): ConvertKitValidator
    {
        $this->flags = $this->flags | self::FLAG__REQUIRE_USERNAME;

        return $this;
    }
}
