<?php

namespace geosim\core;
use apexl\entityCore\Entity;
use apexl\Vault\Vault;
use geosim\core\interfaces\platformInterface;
use geosim\core\platforms\platform;
use geosim\simentity\sim as simEntity;

/**
 * Wrapper class to replace the existing Geosim class with as minimal disruption as possible.
 * As such, some of this is less than ideal...
 * Class sim
 * @package geosim\core
 */
class sim extends Entity {

    /** @var platform */
    public $operator;
    protected $vault;
    protected $lastCall;
    protected $loadFromLocal = TRUE;
    protected $forcePlatform;
    /** @var simEntity */
    public $sim;

    public function __construct()
    {
        parent::__construct();
        $this->vault = Vault::getInstance();
        $this->sim = new simEntity();
    }

    public function getEntityType()
    {
        return __CLASS__;
    }

    /**
     * Allow us to force a specific platform if the sim is not available locally.
     * @param $platform
     * @return $this
     */
    public function forcePlatform($platform){
        $this->sim->platform = $platform;
        $this->forcePlatform = $platform;
        return $this;
    }

    public function forceAccount($account){
        $this->sim->geosimAccount = $account;
        return $this;
    }

    public function resetSim(){
        $this->sim = new simEntity();
        return $this;
    }

    public function setLoadFromLocal($lfl){
        $this->loadFromLocal = $lfl;
        return $this;
    }

    private function mapSimCall($name){
        switch($name){
            case 'get_customer_balance':
                $name = 'getBalance';
                break;
            case 'query_sim_details_iccid':
                $name = 'getSimDetailsICCID';
                break;
            case 'apply_customer_credit':
                $name = 'topUp';
                break;
            case 'apply_customer_debit':
                $name = 'deductCredit';
                break;
            case 'activate_customer':
                $name = 'activate';
                break;
            case 'get_customer_call_history':
                $name = 'getHistory';
                break;
            case 'set_customer_details':
                $name = 'setCustomerDetails';
                break;
            case 'set_subscriber_details':
                $name = 'setSubscriberDetails';
                break;
            case 'set_customer_terms':
                $name = 'setCustomerTerms';
                break;
            case 'get_subscriber_directory_numbers':
                $name = 'getSubscriberDirectoryNumbers';
                break;
            case 'set_subscriber_cli':
                $name = 'setDirectoryNumber';
                break;
            case 'suspend_customer':
                $name = 'suspendCustomer';
                break;
            case 'locate_subscriber':
                $name = 'locateSim';
                break;
            case 'get_customer_status':
                $name = 'getCustomerStatus';
                break;
            case 'send_sms':
                $name = 'sendSMS';
                break;
            case 'get_usage':
                $name = 'getUsage';
                break;
        }

        return $name;
    }

    public function getSimInfo($id){
        if($this->loadFromLocal) {
            //empty ID? wipe the sim info.
            if (empty($id)) {
                $this->sim = new simEntity();
                if($this->forcePlatform){
                    $this->sim->platform = $this->forcePlatform;
                }
                return $this;
            }
            //More than 19? This is an iccid with a checksum. Strip the checksum off. This deals with C9 returning ICCIDs without checksums.
            if (strlen($id) > 19) {
                $id = substr($id, 0, 19);
            }
            if (!empty($id)) {
                $this->sim->load($id);
            }
            if (!isset($this->sim->platform) || empty($this->sim->platform) || !isset($this->sim->msisdn) || empty($this->sim->msisdn)) {
                $this->testForPlatform($id);
            }
        }

        return $this;
    }

    protected function getSimDataByIccid($iccid){
        return $this->sim->load($iccid);
    }

    protected function testForPlatform($id){
        //loop over each available platform and check the credentials.
        foreach($this->config->geosim as $platform => $config){
            //loop over each account available for this platform and test if we can make a request,
            foreach($config->accounts as $account => $pass){
                $this->setPlatform($platform, $account);
                //blindGuess what we're dealing with, so set both to the same ID and test if a call works.
                $this->operator->setSimByICCID($id);
                $this->operator->setSimByMSISDN($id);

                $responseMSISDN = $this->operator->getSimDetailsMSISDN();
                $responseICCID = $this->operator->getSimDetailsICCID();
                //we have a valid response? return out of the loop.
                if(isset($responseMSISDN['iccid']) || isset($responseMSISDN['Sim']['IccId'])){
                    $this->storePlatformInformation($responseMSISDN, $account, $platform);
                    return;
                }
                if(isset($responseICCID['iccid']) || isset($responseICCID['Sim']['IccId'])){
                    $this->storePlatformInformation($responseICCID, $account, $platform);
                    return;
                }
            }
        }
    }

    protected function updateLocalSimWithMSISDN($response){
        if(isset($response['Error']) || isset($response['error'])){
            $this->sim->msisdn = isset($response['Error']['msisdn']) ? $response['Error']['msisdn'] : $response['error']['msisdn'];
        } else {
            if(isset($response['Wire9_data'])) {
                $this->sim->msisdn = $response['Wire9_data']['STATUS_Response']['MSISDN'];
            } else {
                $this->sim->msisdn = $response['STATUS_Response']['MSISDN'];
            }
        }
        $this->sim->store();
    }

    function storePlatformInformation( $response, $account = '', $platform = '')
    {
        //todo, something platform specific? or maybe a unified response object.
        $msisdn = isset($response['identities']['identity']['primary-msisdn']) ? $response['identities']['identity']['primary-msisdn'] : FALSE;
        //False? then check for msisdn from cloud9.
        $msisdn = $msisdn === FALSE && isset($response['Sim']['PublicNumber']) ? $response['Sim']['PublicNumber'] : $msisdn;
        if ($platform == 'jasper'){
            $msisdn = $response['msisdn'];
            $this->sim->active = $response['status'] == "ACTIVATED" ? 1 : 0;
            $this->sim->status = $response['status'];
        }

        //set sim data
        $this->sim->iccid = isset($response['iccid']) ? $response['iccid'] : $response['Sim']['IccId'];
        $this->sim->lastChecked = time();
        $this->sim->geosimAccount = $account;
        $this->sim->platform = $platform;

        $this->sim->msisdn = $msisdn !== FALSE ? $msisdn : '';
        //$this->sim->userId = isset($response['Sim']['UserId']) ? $response['Sim']['UserId'] : 0;
        $this->sim->customerId = isset($response['Sim']['CustomerId']) ? $response['Sim']['CustomerId'] : 0;
        $this->sim->barcode = isset($response['accountCustom2']) && $platform == 'jasper' ? $response['accountCustom2'] : '';
        $this->sim->store();

        return $this;
    }

    protected function setPlatform($pName, $account){

        $platform = "\\geosim\\core\\platforms\\".$pName;
        $this->setOperator($platform::getInstance());
        $this->operator->setApiUrl($this->config->geosim->{$pName}->apiUrl, $this->config->geosim->{$pName}->apiPath);
        $this->operator->setAuth($account, $this->config->geosim->{$pName}->accounts->{$account});
        if(isset($this->config->geosim->{$pName}->apiSchema) && !empty($this->config->geosim->{$pName}->apiSchema)){
            $this->operator->setSchema($this->config->geosim->{$pName}->apiSchema);
        }
        if(!empty($this->sim->data)){
            $this->operator->setSimInfo($this->sim);
            //@todo remove these and amend references to use simInfo.
        }
    }

    public function getPlatform(){
        return isset($this->sim->platform) ? $this->sim->platform : FALSE ;
    }

    public function __call($name, $arguments)
    {
        //first, map the existing call to the correct methods.
        $method = $this->mapSimCall($name);
        //load the sim info, set the appropriate platform.
        $this->checkSimInfoByArgs($arguments);
        if(isset($this->sim->platform) && !empty($this->sim->platform)){
            $this->lastCall = $method;
            //make sure we set the platform and data for this run.
            $this->setPlatform($this->sim->platform, $this->sim->geosimAccount);
            //make sure we set MSISDN and ICCID info for this run.
            //Make the requested call.
            $response = parent::__call($method, $arguments);
            //log the response for the request
            $this->setData($this->operator->mapResponse($response));
            if(isset($response['UpdateLocalRecord'])){
                $this->updateLocalSimWithMSISDN($response);
            }

            //account for C9 activation
            if(isset($this->sim->iccid)){
                $this->getSimInfo($this->sim->iccid);
            } else {
                $this->checkSimInfoByArgs($arguments);
            }
            return $this->data;
        }
    }

    public function getRequestStack()
    {
        if (method_exists($this->operator, 'getRequestStack')) {
            return $this->operator->getRequestStack();
        }

        return null;
    }

    protected function checkSimInfoByArgs(&$arguments){
        $id = isset($arguments[0]) ? $arguments[0] : FALSE;
        $this->getSimInfo($id);
        //The ID worked, so lets unset it from the args list.
        if(isset($this->sim->iccid) && !empty($this->sim->platform)){
            unset($arguments[0]);
        }
    }

    //method to skip the normal __call process.
    //C9 methods
    public function transferAgentCredit($agentId, $dealerId, $amount){
        error_log('transferAgentCredit: -- START');
        $this->setPlatform('cloud9', 'geov5');
        $response = parent::__call('transferAgentCredit', [$agentId, $dealerId, $amount]);
        error_log('transferAgentCredit: -- END '. json_encode($response));
        $this->setData($this->operator->mapResponse($response));
        return $this->data;
    }

    public function GetAgentBalance($agentId){
        $this->setPlatform('cloud9', 'geov5');
        $response = parent::__call('GetAgentBalance', [$agentId]);
        $this->setData($this->operator->mapResponse($response));
        return $this->data;
    }
}